/*:
 * @target MZ
 * @plugindesc v1.0 メモ欄 <OrderId:n> で並び順だけ制御（スキル/アイテム）。他機能は不変・区分も維持。
 * @author You
 *
 * @help OrderIdSortMZ.js
 * スキルやアイテム等のメモ欄に <OrderId:x> と書くと、その数値を「並び順キー」として扱います。
 * 影響は「表示順」だけで、使用可否や費用、効果などの機能には一切触れません。
 *
 * 対応ウィンドウ：
 *  - Window_SkillList（メニュー/戦闘のスキル選択）
 *  - Window_ItemList（メニュー/戦闘/ショップのアイテム選択）
 *
 * 互換：
 *  - 既存の「リスト構築系プラグイン（例：未習得スキル表示）」の後段で
 *    _data が組まれたあとに「安定ソート」を 1 回だけ行います。
 *  - v1.7系プラグイン（UpcomingSkillsListMZ）の「上=既習得／下=未習得」区分を検知し、
 *    その区分を保ったまま各区分内で <OrderId> 順に並べ替えます。
 *
 * 使い方：
 *  - スキル/アイテム/武器/防具のメモ欄に <OrderId:数値> を記述。
 *  - 数値が小さいほど先頭へ。未指定は末尾（デフォルト999999）。
 *
 * 注意：
 *  - 並び順だけを変えます。他プラグインの使用可否フラグ等はそのままです。
 */

(() => {
  'use strict';

  // ========= 共通ユーティリティ =========
  const INF = 999999;

  function noteOrderId(obj) {
    if (!obj || !obj.meta) return null;
    // 大文字小文字ゆるく拾う
    const v = obj.meta.OrderId ?? obj.meta.orderId ?? obj.meta.ORDERID;
    if (v == null) return null;
    const n = Number(v);
    return Number.isFinite(n) ? n : null;
  }

  // 区分キーの取得（UpcomingSkillsListMZ などの「未習得フラグ」を尊重）
  function sectionKey(win, item) {
    // 未習得側を 1、既習得側を 0 にして、既習得→未習得の順を維持
    // 該当フラグが無いウィンドウでは 0 固定（=全体で並べ替え）
    try {
      if (win && win._upcomingFlags && item && item.id != null) {
        return win._upcomingFlags[item.id] ? 1 : 0;
      }
    } catch (_) {}
    return 0;
  }

  // _data を <OrderId> で「区分を保ったまま安定ソート」
  function applyOrderIdSort(win) {
    const data = win && win._data;
    if (!Array.isArray(data) || data.length < 2) return;

    const wrapped = data.map((it, idx) => {
      const sec = sectionKey(win, it);
      const ord = noteOrderId(it);
      return {
        item: it,
        idx,
        sec,
        ord: (ord != null ? ord : INF)
      };
    });

    wrapped.sort((a, b) => {
      // 1) 区分（既習得0 → 未習得1）をまず尊重
      if (a.sec !== b.sec) return a.sec - b.sec;
      // 2) OrderId 昇順
      if (a.ord !== b.ord) return a.ord - b.ord;
      // 3) 元の相対順（安定ソート）
      return a.idx - b.idx;
    });

    for (let i = 0; i < wrapped.length; i++) {
      data[i] = wrapped[i].item;
    }
  }

  // ========= SkillListへの差し込み =========
  const _Skill_makeItemList = Window_SkillList.prototype.makeItemList;
  Window_SkillList.prototype.makeItemList = function() {
    _Skill_makeItemList.call(this);
    applyOrderIdSort(this);
  };

  // ========= ItemListへの差し込み =========
  const _Item_makeItemList = Window_ItemList.prototype.makeItemList;
  Window_ItemList.prototype.makeItemList = function() {
    _Item_makeItemList.call(this);
    applyOrderIdSort(this);
  };

})();
